<?php
/**
 * Plugin Name: Latest Posts Grid
 * Plugin URI: https://digitalhubit.com/latest-posts-grid
 * Description: A simple plugin to display latest posts in a grid with customizable options via shortcode and admin dashboard.
 * Version: 1.6
 * Author: Tariqul Islam
 * Author URI: https://digitalhubit.com
 * Text Domain: latest-posts-grid
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('LPG_VERSION', '1.6');
define('LPG_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LPG_PLUGIN_PATH', plugin_dir_path(__FILE__));

class Latest_Posts_Grid {

    /**
     * Plugin version
     */
    private $version = LPG_VERSION;

    /**
     * Constructor
     */
    public function __construct() {
        // Register shortcode
        add_shortcode('latest_posts_grid', array($this, 'latest_posts_grid_shortcode'));
        
        // Enqueue styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_styles'));
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // Register settings
        add_action('admin_init', array($this, 'register_settings'));
        
        // Enqueue admin scripts and styles
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Register AJAX handlers for admin
        add_action('wp_ajax_lpg_save_shortcode', array($this, 'ajax_save_shortcode'));
        add_action('wp_ajax_lpg_load_shortcode', array($this, 'ajax_load_shortcode'));
        add_action('wp_ajax_lpg_delete_shortcode', array($this, 'ajax_delete_shortcode'));
        
        // Add admin notices
        add_action('admin_notices', array($this, 'admin_notices'));
        
        // Activation and deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Add plugin action links
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'plugin_action_links'));
    }

    /**
     * Enqueue frontend styles
     */
    public function enqueue_frontend_styles() {
        wp_enqueue_style(
            'latest-posts-grid-style',
            LPG_PLUGIN_URL . 'css/style.css',
            array(),
            $this->version
        );
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Latest Posts Grid', 'latest-posts-grid'),
            __('Posts Grid', 'latest-posts-grid'),
            'manage_options',
            'latest-posts-grid',
            array($this, 'admin_page'),
            'dashicons-grid-view',
            30
        );
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('lpg_settings_group', 'lpg_saved_shortcodes', array(
            'sanitize_callback' => array($this, 'sanitize_shortcodes'),
        ));
    }
    
    /**
     * Sanitize shortcodes data
     */
    public function sanitize_shortcodes($input) {
        if (!is_array($input)) {
            return array();
        }
        
        $sanitized = array();
        foreach ($input as $key => $shortcode) {
            if (is_array($shortcode)) {
                $sanitized[sanitize_key($key)] = array_map('sanitize_text_field', $shortcode);
            }
        }
        
        return $sanitized;
    }
    
    /**
     * Enqueue admin scripts and styles - SECURE VERSION
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our plugin page
        if ($hook !== 'toplevel_page_latest-posts-grid') {
            return;
        }
        
        // Enqueue jQuery and jQuery UI with proper dependencies
        wp_enqueue_script('jquery');
        wp_enqueue_script('jquery-ui-core');
        wp_enqueue_script('jquery-ui-tabs');
        wp_enqueue_script('jquery-ui-sortable');
        
        // Enqueue WordPress color picker
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        // Use WordPress's built-in jQuery UI CSS instead of external CDN
        wp_enqueue_style('jquery-ui-theme', includes_url('css/jquery-ui-dialog.min.css'), array(), get_bloginfo('version'));
        
        // Enqueue our admin script with proper dependencies
        wp_enqueue_script(
            'latest-posts-grid-admin',
            LPG_PLUGIN_URL . 'js/admin.js',
            array('jquery', 'jquery-ui-core', 'jquery-ui-tabs', 'jquery-ui-sortable', 'wp-color-picker'),
            $this->version,
            true
        );
        
        // Enqueue our admin styles
        wp_enqueue_style(
            'latest-posts-grid-admin-style',
            LPG_PLUGIN_URL . 'css/admin.css',
            array('wp-color-picker'),
            $this->version
        );
        
        // Add Ajax URL and nonce to script
        wp_localize_script(
            'latest-posts-grid-admin',
            'lpg_admin',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('lpg_admin_nonce'),
                'debug' => defined('WP_DEBUG') && WP_DEBUG,
                'version' => $this->version,
                'strings' => array(
                    'save_success' => __('Grid saved successfully!', 'latest-posts-grid'),
                    'delete_confirm' => __('Are you sure you want to delete this grid? This action cannot be undone.', 'latest-posts-grid'),
                    'copy_success' => __('Shortcode copied to clipboard!', 'latest-posts-grid'),
                    'error_occurred' => __('An error occurred. Please try again.', 'latest-posts-grid'),
                )
            )
        );
    }
    
    /**
     * Admin page content - SECURE VERSION
     */
    public function admin_page() {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'latest-posts-grid'));
        }
        
        // Get categories
        $categories = get_categories(array(
            'orderby' => 'name',
            'order' => 'ASC',
            'hide_empty' => false
        ));
        
        // Get saved shortcodes
        $saved_shortcodes = get_option('lpg_saved_shortcodes', array());
        ?>
        <div class="wrap lpg-admin-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div id="lpg-admin-tabs" class="lpg-admin-tabs">
                <ul class="lpg-tab-nav">
                    <li><a href="#lpg-tabs-1"><?php _e('Create Grid', 'latest-posts-grid'); ?></a></li>
                    <li><a href="#lpg-tabs-2"><?php _e('Saved Grids', 'latest-posts-grid'); ?></a></li>
                    <li><a href="#lpg-tabs-3"><?php _e('How to Use', 'latest-posts-grid'); ?></a></li>
                </ul>
                
                <div id="lpg-tabs-1" class="lpg-tab-content">
                    <div class="lpg-admin-container">
                        <h2><?php _e('Create New Grid', 'latest-posts-grid'); ?></h2>
                        <form id="lpg-shortcode-builder" class="lpg-form">
                            <div class="lpg-form-row">
                                <label for="lpg-shortcode-name"><?php _e('Shortcode Name:', 'latest-posts-grid'); ?></label>
                                <input type="text" id="lpg-shortcode-name" name="shortcode_name" placeholder="<?php esc_attr_e('My Grid', 'latest-posts-grid'); ?>" required maxlength="100">
                                <p class="description"><?php _e('Give your grid a descriptive name for easy identification.', 'latest-posts-grid'); ?></p>
                            </div>
                            
                            <div class="lpg-form-row">
                                <label><?php _e('Select Categories:', 'latest-posts-grid'); ?></label>
                                <div class="lpg-categories-container">
                                    <?php if (!empty($categories)) : ?>
                                        <?php foreach ($categories as $category) : ?>
                                        <label class="lpg-category-checkbox">
                                            <input type="checkbox" name="categories[]" value="<?php echo esc_attr($category->term_id); ?>">
                                            <?php echo esc_html($category->name); ?> 
                                            <span class="lpg-count">(<?php echo esc_html($category->count); ?>)</span>
                                        </label>
                                        <?php endforeach; ?>
                                    <?php else : ?>
                                        <p><?php _e('No categories found. Please create some categories first.', 'latest-posts-grid'); ?></p>
                                    <?php endif; ?>
                                </div>
                                <p class="description"><?php _e('Leave unchecked to include posts from all categories.', 'latest-posts-grid'); ?></p>
                            </div>
                            
                            <div class="lpg-form-row lpg-row-columns">
                                <div class="lpg-column">
                                    <label for="lpg-show-update-info"><?php _e('Display Update Info:', 'latest-posts-grid'); ?></label>
                                    <select id="lpg-show-update-info" name="show_update_info">
                                        <option value="1" selected><?php _e('Yes', 'latest-posts-grid'); ?></option>
                                        <option value="0"><?php _e('No', 'latest-posts-grid'); ?></option>
                                    </select>
                                    <p class="description"><?php _e('Show publication date and author.', 'latest-posts-grid'); ?></p>
                                </div>
                                
                                <div class="lpg-column">
                                    <label for="lpg-title-capitalization"><?php _e('Title Capitalization:', 'latest-posts-grid'); ?></label>
                                    <select id="lpg-title-capitalization" name="title_capitalization">
                                        <option value="default" selected><?php _e('Default', 'latest-posts-grid'); ?></option>
                                        <option value="uppercase"><?php _e('UPPERCASE', 'latest-posts-grid'); ?></option>
                                        <option value="lowercase"><?php _e('lowercase', 'latest-posts-grid'); ?></option>
                                        <option value="capitalize"><?php _e('Title Case', 'latest-posts-grid'); ?></option>
                                        <option value="sentence"><?php _e('Sentence case', 'latest-posts-grid'); ?></option>
                                    </select>
                                    <p class="description"><?php _e('Choose how post titles should be displayed.', 'latest-posts-grid'); ?></p>
                                </div>
                            </div>
                            
                            <div class="lpg-form-row lpg-row-columns">
                                <div class="lpg-column">
                                    <label for="lpg-columns"><?php _e('Number of Columns:', 'latest-posts-grid'); ?></label>
                                    <select id="lpg-columns" name="columns">
                                        <?php for ($i = 1; $i <= 6; $i++) : ?>
                                        <option value="<?php echo $i; ?>" <?php echo ($i == 3) ? 'selected' : ''; ?>><?php echo $i; ?></option>
                                        <?php endfor; ?>
                                    </select>
                                    <p class="description"><?php _e('Grid will be responsive on mobile devices.', 'latest-posts-grid'); ?></p>
                                </div>
                                
                                <div class="lpg-column">
                                    <label for="lpg-posts-per-page"><?php _e('Posts Per Page:', 'latest-posts-grid'); ?></label>
                                    <input type="number" id="lpg-posts-per-page" name="posts_per_page" min="1" max="100" value="6">
                                    <p class="description"><?php _e('Maximum 100 posts per page.', 'latest-posts-grid'); ?></p>
                                </div>
                            </div>
                            
                            <div class="lpg-form-row lpg-row-columns">
                                <div class="lpg-column">
                                    <label for="lpg-order"><?php _e('Order:', 'latest-posts-grid'); ?></label>
                                    <select id="lpg-order" name="order">
                                        <option value="DESC" selected><?php _e('Newest First', 'latest-posts-grid'); ?></option>
                                        <option value="ASC"><?php _e('Oldest First', 'latest-posts-grid'); ?></option>
                                    </select>
                                </div>
                                
                                <div class="lpg-column">
                                    <label for="lpg-orderby"><?php _e('Order By:', 'latest-posts-grid'); ?></label>
                                    <select id="lpg-orderby" name="orderby">
                                        <option value="date" selected><?php _e('Date', 'latest-posts-grid'); ?></option>
                                        <option value="title"><?php _e('Title', 'latest-posts-grid'); ?></option>
                                        <option value="modified"><?php _e('Last Modified', 'latest-posts-grid'); ?></option>
                                        <option value="comment_count"><?php _e('Comment Count', 'latest-posts-grid'); ?></option>
                                        <option value="rand"><?php _e('Random', 'latest-posts-grid'); ?></option>
                                    </select>
                                </div>
                            </div>
                            
                            <!-- Featured Image Control -->
                            <div class="lpg-form-row lpg-row-columns">
                                <div class="lpg-column">
                                    <label for="lpg-show-featured-image"><?php _e('Show Featured Image:', 'latest-posts-grid'); ?></label>
                                    <select id="lpg-show-featured-image" name="show_featured_image">
                                        <option value="1" selected><?php _e('Yes', 'latest-posts-grid'); ?></option>
                                        <option value="0"><?php _e('No', 'latest-posts-grid'); ?></option>
                                    </select>
                                    <p class="description"><?php _e('Display post featured images in the grid.', 'latest-posts-grid'); ?></p>
                                </div>
                                
                                <div class="lpg-column">
                                    <!-- Placeholder for alignment -->
                                </div>
                            </div>
                            
                            <!-- Button Color Settings -->
                            <div class="lpg-form-row lpg-row-columns">
                                <div class="lpg-column">
                                    <label for="lpg-button-color"><?php _e('Button Color:', 'latest-posts-grid'); ?></label>
                                    <input type="text" id="lpg-button-color" name="button_color" value="#4299e1" class="lpg-color-picker">
                                    <p class="description"><?php _e('Color of the "Read More" button.', 'latest-posts-grid'); ?></p>
                                </div>
                                
                                <div class="lpg-column">
                                    <label for="lpg-button-hover-color"><?php _e('Button Hover Color:', 'latest-posts-grid'); ?></label>
                                    <input type="text" id="lpg-button-hover-color" name="button_hover_color" value="#3182ce" class="lpg-color-picker">
                                    <p class="description"><?php _e('Color when hovering over the button.', 'latest-posts-grid'); ?></p>
                                </div>
                            </div>
                            
                            <!-- Pagination Color Settings -->
                            <div class="lpg-form-row lpg-row-columns">
                                <div class="lpg-column">
                                    <label for="lpg-pagination-bg-color"><?php _e('Pagination Background:', 'latest-posts-grid'); ?></label>
                                    <input type="text" id="lpg-pagination-bg-color" name="pagination_bg_color" value="#ffffff" class="lpg-color-picker">
                                    <p class="description"><?php _e('Background color of pagination buttons.', 'latest-posts-grid'); ?></p>
                                </div>
                                
                                <div class="lpg-column">
                                    <label for="lpg-pagination-text-color"><?php _e('Pagination Text Color:', 'latest-posts-grid'); ?></label>
                                    <input type="text" id="lpg-pagination-text-color" name="pagination_text_color" value="#4a5568" class="lpg-color-picker">
                                    <p class="description"><?php _e('Text color of pagination numbers.', 'latest-posts-grid'); ?></p>
                                </div>
                            </div>
                            
                            <div class="lpg-form-row lpg-row-columns">
                                <div class="lpg-column">
                                    <label for="lpg-pagination-active-color"><?php _e('Active Page Color:', 'latest-posts-grid'); ?></label>
                                    <input type="text" id="lpg-pagination-active-color" name="pagination_active_color" value="#4299e1" class="lpg-color-picker">
                                    <p class="description"><?php _e('Color of the current page button.', 'latest-posts-grid'); ?></p>
                                </div>
                                
                                <div class="lpg-column">
                                    <label for="lpg-pagination-hover-color"><?php _e('Pagination Hover Color:', 'latest-posts-grid'); ?></label>
                                    <input type="text" id="lpg-pagination-hover-color" name="pagination_hover_color" value="#f7fafc" class="lpg-color-picker">
                                    <p class="description"><?php _e('Background color when hovering over pagination.', 'latest-posts-grid'); ?></p>
                                </div>
                            </div>
                            
                            <!-- Pagination settings -->
                            <div class="lpg-form-row lpg-row-columns">
                                <div class="lpg-column">
                                    <label for="lpg-pagination"><?php _e('Enable Pagination:', 'latest-posts-grid'); ?></label>
                                    <select id="lpg-pagination" name="pagination">
                                        <option value="0"><?php _e('No', 'latest-posts-grid'); ?></option>
                                        <option value="1" selected><?php _e('Yes', 'latest-posts-grid'); ?></option>
                                    </select>
                                    <p class="description"><?php _e('Allow users to navigate through multiple pages.', 'latest-posts-grid'); ?></p>
                                </div>
                                
                                <div class="lpg-column">
                                    <label for="lpg-pagination-type"><?php _e('Pagination Type:', 'latest-posts-grid'); ?></label>
                                    <select id="lpg-pagination-type" name="pagination_type">
                                        <option value="numbers"><?php _e('Numbers Only', 'latest-posts-grid'); ?></option>
                                        <option value="prev_next"><?php _e('Previous/Next Only', 'latest-posts-grid'); ?></option>
                                        <option value="both" selected><?php _e('Numbers with Next Button (1,2,3.....121 Next →)', 'latest-posts-grid'); ?></option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="lpg-form-row">
                                <button type="submit" class="button button-primary"><?php _e('Save Grid', 'latest-posts-grid'); ?></button>
                                <div id="lpg-shortcode-preview" class="lpg-shortcode-preview"></div>
                            </div>
                        </form>
                    </div>
                </div>
                
                <div id="lpg-tabs-2" class="lpg-tab-content">
                    <div class="lpg-admin-container">
                        <h2><?php _e('Saved Grids', 'latest-posts-grid'); ?></h2>
                        <?php if (empty($saved_shortcodes)) : ?>
                            <p><?php _e('No saved grids yet. Create one in the "Create Grid" tab.', 'latest-posts-grid'); ?></p>
                        <?php else : ?>
                            <table class="widefat lpg-shortcodes-table">
                                <thead>
                                    <tr>
                                        <th><?php _e('Name', 'latest-posts-grid'); ?></th>
                                        <th><?php _e('Shortcode', 'latest-posts-grid'); ?></th>
                                        <th><?php _e('Categories', 'latest-posts-grid'); ?></th>
                                        <th><?php _e('Settings', 'latest-posts-grid'); ?></th>
                                        <th><?php _e('Actions', 'latest-posts-grid'); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($saved_shortcodes as $id => $shortcode) : ?>
                                    <tr data-id="<?php echo esc_attr($id); ?>">
                                        <td><?php echo esc_html($shortcode['name']); ?></td>
                                        <td>
                                            <code class="lpg-shortcode-code">[latest_posts_grid id="<?php echo esc_attr($id); ?>"]</code>
                                            <button type="button" class="button button-small lpg-copy-shortcode" data-shortcode='[latest_posts_grid id="<?php echo esc_attr($id); ?>"]' title="<?php esc_attr_e('Copy to clipboard', 'latest-posts-grid'); ?>">
                                                <span class="dashicons dashicons-clipboard"></span>
                                            </button>
                                        </td>
                                        <td>
                                            <?php 
                                            if (!empty($shortcode['categories'])) {
                                                $cat_names = array();
                                                foreach ($shortcode['categories'] as $cat_id) {
                                                    $cat = get_category($cat_id);
                                                    if ($cat && !is_wp_error($cat)) {
                                                        $cat_names[] = $cat->name;
                                                    }
                                                }
                                                echo esc_html(implode(', ', $cat_names));
                                            } else {
                                                _e('All Categories', 'latest-posts-grid');
                                            }
                                            ?>
                                        </td>
                                        <td>
                                            <span class="lpg-setting"><?php _e('Columns:', 'latest-posts-grid'); ?> <?php echo esc_html($shortcode['columns']); ?></span>
                                            <span class="lpg-setting"><?php _e('Posts:', 'latest-posts-grid'); ?> <?php echo esc_html($shortcode['posts_per_page']); ?></span>
                                            <span class="lpg-setting"><?php _e('Order:', 'latest-posts-grid'); ?> <?php echo esc_html($shortcode['order']); ?></span>
                                            <span class="lpg-setting"><?php _e('Update Info:', 'latest-posts-grid'); ?> <?php echo isset($shortcode['show_update_info']) && $shortcode['show_update_info'] ? __('Yes', 'latest-posts-grid') : __('No', 'latest-posts-grid'); ?></span>
                                            <span class="lpg-setting"><?php _e('Title Style:', 'latest-posts-grid'); ?> <?php echo isset($shortcode['title_capitalization']) ? esc_html(ucfirst($shortcode['title_capitalization'])) : __('Default', 'latest-posts-grid'); ?></span>
                                            <span class="lpg-setting"><?php _e('Featured Image:', 'latest-posts-grid'); ?> <?php echo isset($shortcode['show_featured_image']) && $shortcode['show_featured_image'] ? __('Yes', 'latest-posts-grid') : __('No', 'latest-posts-grid'); ?></span>
                                            <span class="lpg-setting"><?php _e('Pagination:', 'latest-posts-grid'); ?> <?php echo isset($shortcode['pagination']) && $shortcode['pagination'] ? __('Yes', 'latest-posts-grid') : __('No', 'latest-posts-grid'); ?></span>
                                            <?php if (isset($shortcode['button_color']) && !empty($shortcode['button_color'])) : 
                                                $button_color = sanitize_hex_color($shortcode['button_color']);
                                                if ($button_color) : ?>
                                                <span class="lpg-setting"><?php _e('Button:', 'latest-posts-grid'); ?> <span class="lpg-color-preview" style="background-color: <?php echo esc_attr($button_color); ?>" title="<?php echo esc_attr($button_color); ?>"></span></span>
                                            <?php endif; endif; ?>
                                            <?php if (isset($shortcode['pagination_active_color']) && !empty($shortcode['pagination_active_color'])) :
                                                $pagination_color = sanitize_hex_color($shortcode['pagination_active_color']);
                                                if ($pagination_color) : ?>
                                                <span class="lpg-setting"><?php _e('Pagination:', 'latest-posts-grid'); ?> <span class="lpg-color-preview" style="background-color: <?php echo esc_attr($pagination_color); ?>" title="<?php echo esc_attr($pagination_color); ?>"></span></span>
                                            <?php endif; endif; ?>
                                        </td>
                                        <td>
                                            <button type="button" class="button lpg-edit-shortcode" data-id="<?php echo esc_attr($id); ?>"><?php _e('Edit', 'latest-posts-grid'); ?></button>
                                            <button type="button" class="button lpg-delete-shortcode" data-id="<?php echo esc_attr($id); ?>"><?php _e('Delete', 'latest-posts-grid'); ?></button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php endif; ?>
                    </div>
                </div>
                
                <div id="lpg-tabs-3" class="lpg-tab-content">
                    <div class="lpg-admin-container lpg-help-tab">
                        <h2><?php _e('How to Use Latest Posts Grid', 'latest-posts-grid'); ?></h2>
                        
                        <div class="lpg-help-section">
                            <h3><?php _e('Creating a Grid', 'latest-posts-grid'); ?></h3>
                            <ol>
                                <li><?php _e('Go to the "Create Grid" tab', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Enter a name for your grid', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Select the categories you want to include (leave unchecked for all categories)', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Choose whether to display update information (publication date and author)', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Select title capitalization style', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Choose the number of columns (1-6)', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Set how many posts to display per page', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Select the order and sorting method', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Enable or disable featured images', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Choose button colors for the "Read More" button', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Customize pagination colors and design', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Enable or disable pagination and choose the pagination style', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Click "Save Grid"', 'latest-posts-grid'); ?></li>
                            </ol>
                        </div>
                        
                        <div class="lpg-help-section">
                            <h3><?php _e('Using Your Grid', 'latest-posts-grid'); ?></h3>
                            <p><?php _e('After creating a grid, you can add it to any post or page using the shortcode:', 'latest-posts-grid'); ?></p>
                            <pre>[latest_posts_grid id="your_grid_id"]</pre>
                            <p><?php _e('You can find and copy your grid\'s shortcode from the "Saved Grids" tab.', 'latest-posts-grid'); ?></p>
                        </div>
                        
                        <div class="lpg-help-section">
                            <h3><?php _e('Title Capitalization Options', 'latest-posts-grid'); ?></h3>
                            <ul>
                                <li><strong><?php _e('Default', 'latest-posts-grid'); ?></strong> - <?php _e('Uses the original post title formatting', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('UPPERCASE', 'latest-posts-grid'); ?></strong> - <?php _e('Converts all letters to uppercase', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('lowercase', 'latest-posts-grid'); ?></strong> - <?php _e('Converts all letters to lowercase', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('Title Case', 'latest-posts-grid'); ?></strong> - <?php _e('Capitalizes the first letter of each word', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('Sentence case', 'latest-posts-grid'); ?></strong> - <?php _e('Capitalizes only the first letter of the title', 'latest-posts-grid'); ?></li>
                            </ul>
                        </div>
                        
                        <div class="lpg-help-section">
                            <h3><?php _e('Button Color Customization', 'latest-posts-grid'); ?></h3>
                            <ul>
                                <li><strong><?php _e('Button Color', 'latest-posts-grid'); ?></strong> - <?php _e('Set the default background color of the "Read More" button', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('Button Hover Color', 'latest-posts-grid'); ?></strong> - <?php _e('Set the background color when users hover over the button', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('Color Picker', 'latest-posts-grid'); ?></strong> - <?php _e('Easy-to-use WordPress color picker with preset colors', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('Live Preview', 'latest-posts-grid'); ?></strong> - <?php _e('See how your button colors will look before saving', 'latest-posts-grid'); ?></li>
                            </ul>
                        </div>
                        
                        <div class="lpg-help-section">
                            <h3><?php _e('Pagination Color Customization', 'latest-posts-grid'); ?></h3>
                            <ul>
                                <li><strong><?php _e('Pagination Background', 'latest-posts-grid'); ?></strong> - <?php _e('Sets the background color of pagination buttons', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('Pagination Text Color', 'latest-posts-grid'); ?></strong> - <?php _e('Sets the text color of pagination numbers', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('Active Page Color', 'latest-posts-grid'); ?></strong> - <?php _e('Sets the color of the current/active page button', 'latest-posts-grid'); ?></li>
                                <li><strong><?php _e('Pagination Hover Color', 'latest-posts-grid'); ?></strong> - <?php _e('Sets the background color when hovering over pagination buttons', 'latest-posts-grid'); ?></li>
                            </ul>
                        </div>
                        
                        <div class="lpg-help-section">
                            <h3><?php _e('Support', 'latest-posts-grid'); ?></h3>
                            <p><?php _e('If you encounter any issues:', 'latest-posts-grid'); ?></p>
                            <ul>
                                <li><?php _e('Check your WordPress error logs', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Ensure all plugin files are uploaded correctly', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Try deactivating other plugins to check for conflicts', 'latest-posts-grid'); ?></li>
                                <li><?php _e('Check browser console for JavaScript errors', 'latest-posts-grid'); ?></li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        /* Fallback CSS for tabs if jQuery UI doesn't load */
        .lpg-admin-tabs {
            margin-top: 20px;
        }
        
        .lpg-tab-nav {
            list-style: none;
            margin: 0;
            padding: 0;
            border-bottom: 1px solid #ccc;
            display: flex;
        }
        
        .lpg-tab-nav li {
            margin: 0 2px 0 0;
        }
        
        .lpg-tab-nav a {
            display: block;
            padding: 10px 15px;
            background: #f1f1f1;
            color: #333;
            text-decoration: none;
            border: 1px solid #ccc;
            border-bottom: none;
            border-radius: 4px 4px 0 0;
        }
        
        .lpg-tab-nav .ui-tabs-active a,
        .lpg-tab-nav a:hover {
            background: #fff;
            color: #0073aa;
        }
        
        .lpg-tab-content {
            border: 1px solid #ccc;
            border-top: none;
            padding: 0;
        }
        
        .lpg-tab-content.ui-tabs-hide {
            display: none;
        }
        </style>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Initialize tabs with fallback
            if ($.fn.tabs) {
                try {
                    $('#lpg-admin-tabs').tabs();
                } catch (error) {
                    console.warn('jQuery UI Tabs failed to initialize:', error);
                    initFallbackTabs();
                }
            } else {
                initFallbackTabs();
            }
            
            function initFallbackTabs() {
                $('.lpg-tab-nav a').on('click', function(e) {
                    e.preventDefault();
                    var target = $(this).attr('href');
                    
                    // Hide all tab content
                    $('.lpg-tab-content').hide();
                    $('.lpg-tab-nav li').removeClass('active');
                    
                    // Show selected tab content
                    $(target).show();
                    $(this).parent().addClass('active');
                });
                
                // Show first tab by default
                $('.lpg-tab-content:first').show();
                $('.lpg-tab-nav li:first').addClass('active');
            }
        });
        </script>
        <?php
    }
    
    /**
     * AJAX handler for saving shortcode - ENHANCED SECURITY VERSION
     */
    public function ajax_save_shortcode() {
        // Verify nonce first
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'lpg_admin_nonce')) {
            wp_send_json_error(array(
                'message' => __('Security check failed. Please refresh the page and try again.', 'latest-posts-grid'),
                'code' => 'invalid_nonce'
            ), 403);
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => __('You do not have permission to perform this action.', 'latest-posts-grid'),
                'code' => 'insufficient_permissions'
            ), 403);
        }
        
        // Enhanced input validation with ranges
        $name = isset($_POST['shortcode_name']) ? sanitize_text_field(trim($_POST['shortcode_name'])) : '';
        if (empty($name) || strlen($name) > 100) {
            wp_send_json_error(array(
                'message' => __('Shortcode name is required and must be less than 100 characters.', 'latest-posts-grid'),
                'code' => 'invalid_name'
            ), 400);
        }
        
        // Validate categories array
        $categories = array();
        if (isset($_POST['categories']) && is_array($_POST['categories'])) {
            foreach ($_POST['categories'] as $cat_id) {
                $cat_id = intval($cat_id);
                if ($cat_id > 0 && term_exists($cat_id, 'category')) {
                    $categories[] = $cat_id;
                }
            }
        }
        
        // Validate columns with proper range
        $columns = isset($_POST['columns']) ? intval($_POST['columns']) : 3;
        if ($columns < 1 || $columns > 6) {
            $columns = 3;
        }
        
        // Validate posts per page with reasonable limits
        $posts_per_page = isset($_POST['posts_per_page']) ? intval($_POST['posts_per_page']) : 6;
        if ($posts_per_page < 1 || $posts_per_page > 100) {
            $posts_per_page = 6;
        }
        
        // Validate order parameter
        $order = isset($_POST['order']) && in_array($_POST['order'], array('ASC', 'DESC')) ? $_POST['order'] : 'DESC';
        
        // Validate orderby parameter
        $valid_orderby = array('date', 'title', 'modified', 'comment_count', 'rand', 'menu_order');
        $orderby = isset($_POST['orderby']) && in_array($_POST['orderby'], $valid_orderby) ? $_POST['orderby'] : 'date';
        
        // Validate boolean values
        $show_update_info = isset($_POST['show_update_info']) ? intval($_POST['show_update_info']) : 1;
        $show_update_info = in_array($show_update_info, array(0, 1)) ? $show_update_info : 1;
        
        $show_featured_image = isset($_POST['show_featured_image']) ? intval($_POST['show_featured_image']) : 1;
        $show_featured_image = in_array($show_featured_image, array(0, 1)) ? $show_featured_image : 1;
        
        $pagination = isset($_POST['pagination']) ? intval($_POST['pagination']) : 1;
        $pagination = in_array($pagination, array(0, 1)) ? $pagination : 1;
        
        // Validate title capitalization
        $valid_caps = array('default', 'uppercase', 'lowercase', 'capitalize', 'sentence');
        $title_capitalization = isset($_POST['title_capitalization']) && in_array($_POST['title_capitalization'], $valid_caps) 
            ? $_POST['title_capitalization'] : 'default';
        
        // Validate pagination type
        $valid_pagination_types = array('numbers', 'prev_next', 'both');
        $pagination_type = isset($_POST['pagination_type']) && in_array($_POST['pagination_type'], $valid_pagination_types)
            ? $_POST['pagination_type'] : 'both';
        
        // Enhanced color validation with fallbacks
        $color_fields = array(
            'button_color' => '#4299e1',
            'button_hover_color' => '#3182ce',
            'pagination_bg_color' => '#ffffff',
            'pagination_text_color' => '#4a5568',
            'pagination_active_color' => '#4299e1',
            'pagination_hover_color' => '#f7fafc'
        );
        
        $colors = array();
        foreach ($color_fields as $field => $default) {
            $color = isset($_POST[$field]) ? sanitize_hex_color($_POST[$field]) : '';
            $colors[$field] = !empty($color) ? $color : $default;
        }
        
        // Validate editing ID if provided
        $id = isset($_POST['id']) ? sanitize_key($_POST['id']) : '';
        if (!empty($id) && !preg_match('/^lpg_[0-9]+_[0-9]+$/', $id)) {
            wp_send_json_error(array(
                'message' => __('Invalid grid ID format.', 'latest-posts-grid'),
                'code' => 'invalid_id'
            ), 400);
        }
        
        // Get saved shortcodes
        $saved_shortcodes = get_option('lpg_saved_shortcodes', array());
        if (!is_array($saved_shortcodes)) {
            $saved_shortcodes = array();
        }
        
        // Generate ID if not editing
        if (empty($id)) {
            $id = 'lpg_' . time() . '_' . wp_rand(100, 999);
            
            // Ensure unique ID
            $attempts = 0;
            while (isset($saved_shortcodes[$id]) && $attempts < 10) {
                $id = 'lpg_' . time() . '_' . wp_rand(100, 999);
                $attempts++;
            }
        }
        
        // Prepare shortcode data
        $shortcode_data = array(
            'name' => $name,
            'categories' => $categories,
            'columns' => $columns,
            'posts_per_page' => $posts_per_page,
            'order' => $order,
            'orderby' => $orderby,
            'show_update_info' => $show_update_info,
            'title_capitalization' => $title_capitalization,
            'show_featured_image' => $show_featured_image,
            'pagination' => $pagination,
            'pagination_type' => $pagination_type,
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql')
        );
        
        // Add colors
        foreach ($colors as $field => $value) {
            $shortcode_data[$field] = $value;
        }
        
        // Update existing shortcode's created_at time
        if (isset($saved_shortcodes[$id]) && isset($saved_shortcodes[$id]['created_at'])) {
            $shortcode_data['created_at'] = $saved_shortcodes[$id]['created_at'];
        }
        
        // Save shortcode
        $saved_shortcodes[$id] = $shortcode_data;
        
        if (!update_option('lpg_saved_shortcodes', $saved_shortcodes)) {
            $this->log_error('Failed to save shortcode to database', $shortcode_data);
            wp_send_json_error(array(
                'message' => __('Failed to save grid configuration. Please try again.', 'latest-posts-grid'),
                'code' => 'save_failed'
            ), 500);
        }
        
        // Prepare category names for response
        $cat_names = array();
        if (!empty($categories)) {
            foreach ($categories as $cat_id) {
                $cat = get_category($cat_id);
                if ($cat && !is_wp_error($cat)) {
                    $cat_names[] = $cat->name;
                }
            }
        }
        
        // Log successful save
        $this->log_error('Grid saved successfully: ' . $name . ' (ID: ' . $id . ')');
        
        // Send success response
        wp_send_json_success(array(
            'id' => $id,
            'name' => $name,
            'shortcode' => '[latest_posts_grid id="' . $id . '"]',
            'categories' => !empty($cat_names) ? implode(', ', $cat_names) : __('All Categories', 'latest-posts-grid'),
            'settings' => $shortcode_data,
            'message' => __('Grid saved successfully!', 'latest-posts-grid')
        ));
    }
    
    /**
     * AJAX handler for loading shortcode - ENHANCED VERSION
     */
    public function ajax_load_shortcode() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'lpg_admin_nonce')) {
            wp_send_json_error(array(
                'message' => __('Security check failed.', 'latest-posts-grid'),
                'code' => 'invalid_nonce'
            ), 403);
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => __('Insufficient permissions.', 'latest-posts-grid'),
                'code' => 'insufficient_permissions'
            ), 403);
        }
        
        // Validate shortcode ID
        $id = isset($_POST['id']) ? sanitize_key($_POST['id']) : '';
        if (empty($id)) {
            wp_send_json_error(array(
                'message' => __('Grid ID is required.', 'latest-posts-grid'),
                'code' => 'missing_id'
            ), 400);
        }
        
        // Get saved shortcodes
        $saved_shortcodes = get_option('lpg_saved_shortcodes', array());
        if (!is_array($saved_shortcodes) || !isset($saved_shortcodes[$id])) {
            wp_send_json_error(array(
                'message' => __('Grid not found.', 'latest-posts-grid'),
                'code' => 'shortcode_not_found'
            ), 404);
        }
        
        // Return shortcode data with defaults for missing fields
        $defaults = array(
            'name' => '',
            'categories' => array(),
            'columns' => 3,
            'posts_per_page' => 6,
            'order' => 'DESC',
            'orderby' => 'date',
            'show_update_info' => 1,
            'title_capitalization' => 'default',
            'show_featured_image' => 1,
            'button_color' => '#4299e1',
            'button_hover_color' => '#3182ce',
            'pagination_bg_color' => '#ffffff',
            'pagination_text_color' => '#4a5568',
            'pagination_active_color' => '#4299e1',
            'pagination_hover_color' => '#f7fafc',
            'pagination' => 1,
            'pagination_type' => 'both'
        );
        
        $shortcode_data = wp_parse_args($saved_shortcodes[$id], $defaults);
        
        wp_send_json_success($shortcode_data);
    }
    
    /**
     * AJAX handler for deleting shortcode - ENHANCED VERSION
     */
    public function ajax_delete_shortcode() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'lpg_admin_nonce')) {
            wp_send_json_error(array(
                'message' => __('Security check failed.', 'latest-posts-grid'),
                'code' => 'invalid_nonce'
            ), 403);
        }
        
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array(
                'message' => __('Insufficient permissions.', 'latest-posts-grid'),
                'code' => 'insufficient_permissions'
            ), 403);
        }
        
        // Validate shortcode ID
        $id = isset($_POST['id']) ? sanitize_key($_POST['id']) : '';
        if (empty($id)) {
            wp_send_json_error(array(
                'message' => __('Grid ID is required.', 'latest-posts-grid'),
                'code' => 'missing_id'
            ), 400);
        }
        
        // Get saved shortcodes
        $saved_shortcodes = get_option('lpg_saved_shortcodes', array());
        if (!is_array($saved_shortcodes) || !isset($saved_shortcodes[$id])) {
            wp_send_json_error(array(
                'message' => __('Grid not found.', 'latest-posts-grid'),
                'code' => 'shortcode_not_found'
            ), 404);
        }
        
        // Store name for response and logging
        $shortcode_name = $saved_shortcodes[$id]['name'] ?? __('Unknown', 'latest-posts-grid');
        
        // Delete shortcode
        unset($saved_shortcodes[$id]);
        
        if (!update_option('lpg_saved_shortcodes', $saved_shortcodes)) {
            $this->log_error('Failed to delete shortcode from database: ' . $id);
            wp_send_json_error(array(
                'message' => __('Failed to delete grid. Please try again.', 'latest-posts-grid'),
                'code' => 'delete_failed'
            ), 500);
        }
        
        // Log successful deletion
        $this->log_error('Grid deleted successfully: ' . $shortcode_name . ' (ID: ' . $id . ')');
        
        wp_send_json_success(array(
            'message' => sprintf(__('Grid "%s" deleted successfully!', 'latest-posts-grid'), esc_html($shortcode_name))
        ));
    }
    
    /**
     * Shortcode callback function - ENHANCED ERROR HANDLING
     */
    public function latest_posts_grid_shortcode($atts) {
        // Start output buffering to catch any errors
        ob_start();
        
        try {
            // Parse shortcode attributes
            $atts = shortcode_atts(
                array(
                    'id' => '',
                    'columns' => 3,
                    'posts_per_page' => 6,
                    'categories' => '',
                    'order' => 'DESC',
                    'orderby' => 'date',
                    'show_update_info' => 1,
                    'title_capitalization' => 'default',
                    'show_featured_image' => 1,
                    'button_color' => '#4299e1',
                    'button_hover_color' => '#3182ce',
                    'pagination_bg_color' => '#ffffff',
                    'pagination_text_color' => '#4a5568',
                    'pagination_active_color' => '#4299e1',
                    'pagination_hover_color' => '#f7fafc',
                    'pagination' => 1,
                    'pagination_type' => 'both',
                    'paged' => 1,
                ),
                $atts,
                'latest_posts_grid'
            );
            
            // Check if we're using a saved grid
            if (!empty($atts['id'])) {
                $saved_shortcodes = get_option('lpg_saved_shortcodes', array());
                
                if (isset($saved_shortcodes[$atts['id']])) {
                    $saved = $saved_shortcodes[$atts['id']];
                    
                    // Override with saved settings, ensuring all required fields exist
                    foreach ($saved as $key => $value) {
                        if (isset($atts[$key])) {
                            $atts[$key] = $value;
                        }
                    }
                    
                    // Process categories from saved settings
                    if (!empty($saved['categories']) && is_array($saved['categories'])) {
                        $atts['categories'] = implode(',', $saved['categories']);
                    }
                } else {
                    // Grid not found, show error message
                    ob_end_clean();
                    return '<div class="lpg-error" style="background: #f8d7da; color: #721c24; padding: 15px; border-radius: 4px; border: 1px solid #f5c6cb; margin: 20px 0; text-align: center;">' . 
                           sprintf(__('Error: Grid "%s" not found. Please check your shortcode ID or create a new grid.', 'latest-posts-grid'), esc_html($atts['id'])) . 
                           '</div>';
                }
            }
            
            // Validate and sanitize attributes
            $atts['columns'] = max(1, min(6, intval($atts['columns'])));
            $atts['posts_per_page'] = max(1, min(100, intval($atts['posts_per_page'])));
            $atts['show_update_info'] = intval($atts['show_update_info']);
            $atts['show_featured_image'] = intval($atts['show_featured_image']);
            $atts['pagination'] = intval($atts['pagination']);
            
            // Get current page number
            if (get_query_var('paged')) {
                $atts['paged'] = get_query_var('paged');
            } elseif (get_query_var('page')) {
                $atts['paged'] = get_query_var('page');
            } else {
                $atts['paged'] = 1;
            }
            
            // Generate the grid
            $output = $this->generate_grid($atts);
            
            // Clean any accidental output
            ob_end_clean();
            
            return $output;
            
        } catch (Exception $e) {
            // Clean buffer and log error
            ob_end_clean();
            $this->log_error('Shortcode Error: ' . $e->getMessage(), $atts);
            
            // Return user-friendly error message
            if (current_user_can('manage_options')) {
                return '<div class="lpg-error" style="background: #f8d7da; color: #721c24; padding: 15px; border-radius: 4px; border: 1px solid #f5c6cb; margin: 20px 0; text-align: center;">' . 
                       sprintf(__('Latest Posts Grid Error: %s', 'latest-posts-grid'), esc_html($e->getMessage())) . 
                       '</div>';
            } else {
                return '<div class="lpg-error" style="background: #f8d7da; color: #721c24; padding: 15px; border-radius: 4px; border: 1px solid #f5c6cb; margin: 20px 0; text-align: center;">' . 
                       __('Unable to display posts grid. Please contact the site administrator.', 'latest-posts-grid') . 
                       '</div>';
            }
        }
    }
    
    /**
     * Apply title capitalization
     * 
     * @param string $title The original title
     * @param string $capitalization The capitalization type
     * @return string The formatted title
     */
    private function apply_title_capitalization($title, $capitalization) {
        switch ($capitalization) {
            case 'uppercase':
                return strtoupper($title);
            case 'lowercase':
                return strtolower($title);
            case 'capitalize':
                return ucwords(strtolower($title));
            case 'sentence':
                return ucfirst(strtolower($title));
            case 'default':
            default:
                return $title;
        }
    }
    
    /**
     * Generate the grid HTML - ENHANCED VERSION
     * 
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    private function generate_grid($atts) {
        // Convert columns to integer and validate
        $columns = max(1, min(6, intval($atts['columns'])));

        // Prepare query arguments
        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => intval($atts['posts_per_page']),
            'order' => ($atts['order'] === 'ASC') ? 'ASC' : 'DESC',
            'orderby' => sanitize_key($atts['orderby']),
            'paged' => max(1, intval($atts['paged'])),
            'no_found_rows' => false, // We need this for pagination
        );

        // Add categories parameter if provided
        if (!empty($atts['categories'])) {
            $categories = explode(',', $atts['categories']);
            $cat_ids = array();
            
            foreach ($categories as $cat) {
                $cat = trim($cat);
                if (is_numeric($cat)) {
                    $cat_id = intval($cat);
                    if ($cat_id > 0 && term_exists($cat_id, 'category')) {
                        $cat_ids[] = $cat_id;
                    }
                } else {
                    $term = get_term_by('slug', sanitize_title($cat), 'category');
                    if ($term && !is_wp_error($term)) {
                        $cat_ids[] = $term->term_id;
                    }
                }
            }
            
            if (!empty($cat_ids)) {
                $args['category__in'] = $cat_ids;
            }
        }

        // Get posts
        $query = new WP_Query($args);

        // Start output buffering
        ob_start();
        
        // Get capitalization CSS class
        $title_cap_class = '';
        if ($atts['title_capitalization'] !== 'default') {
            $title_cap_class = ' lpg-title-' . esc_attr($atts['title_capitalization']);
        }
        
        // Generate unique ID for this grid instance
        $grid_id = 'lpg-grid-' . wp_rand(1000, 9999);
        
        // Sanitize colors before using them
        $button_color = sanitize_hex_color($atts['button_color']) ?: '#4299e1';
        $button_hover_color = sanitize_hex_color($atts['button_hover_color']) ?: '#3182ce';
        $pagination_bg_color = sanitize_hex_color($atts['pagination_bg_color']) ?: '#ffffff';
        $pagination_text_color = sanitize_hex_color($atts['pagination_text_color']) ?: '#4a5568';
        $pagination_active_color = sanitize_hex_color($atts['pagination_active_color']) ?: '#4299e1';
        $pagination_hover_color = sanitize_hex_color($atts['pagination_hover_color']) ?: '#f7fafc';
        
        // Add custom CSS for button and pagination colors
        echo '<style>';
        echo '#' . esc_attr($grid_id) . ' .lpg-read-more {';
        echo 'background-color: ' . esc_attr($button_color) . ' !important;';
        echo '}';
        echo '#' . esc_attr($grid_id) . ' .lpg-read-more:hover {';
        echo 'background-color: ' . esc_attr($button_hover_color) . ' !important;';
        echo '}';
        
        // Pagination colors
        echo '#' . esc_attr($grid_id) . ' .lpg-pagination .page-numbers {';
        echo 'background-color: ' . esc_attr($pagination_bg_color) . ' !important;';
        echo 'color: ' . esc_attr($pagination_text_color) . ' !important;';
        echo '}';
        echo '#' . esc_attr($grid_id) . ' .lpg-pagination .page-numbers.current {';
        echo 'background-color: ' . esc_attr($pagination_active_color) . ' !important;';
        echo 'color: #fff !important;';
        echo 'border-color: ' . esc_attr($pagination_active_color) . ' !important;';
        echo '}';
        echo '#' . esc_attr($grid_id) . ' .lpg-pagination .page-numbers:hover:not(.current):not(.dots):not(.comma) {';
        echo 'background-color: ' . esc_attr($pagination_hover_color) . ' !important;';
        echo '}';
        echo '</style>';
        
        echo '<div class="lpg-container" id="' . esc_attr($grid_id) . '">';
        echo '<div class="lpg-grid lpg-columns-' . esc_attr($columns) . '">';

        if ($query->have_posts()) {
            // Loop through posts
            while ($query->have_posts()) {
                $query->the_post();
                
                // Post container
                echo '<div class="lpg-grid-item">';
                
                // Featured image (if available and enabled)
                if (has_post_thumbnail() && $atts['show_featured_image']) {
                    echo '<div class="lpg-featured-image">';
                    the_post_thumbnail('medium', array('loading' => 'lazy'));
                    echo '</div>';
                }
                
                echo '<div class="lpg-grid-item-content">';
                
                // Get and format the title
                $original_title = get_the_title();
                $formatted_title = $this->apply_title_capitalization($original_title, $atts['title_capitalization']);
                
                // Post title with link and capitalization class
                echo '<h3 class="lpg-title' . $title_cap_class . '"><a href="' . esc_url(get_permalink()) . '">' . esc_html($formatted_title) . '</a></h3>';
                
                // Update information in "March 29, 2025 by Author" format
                if ($atts['show_update_info']) {
                    echo '<div class="lpg-update-info">' . 
                         sprintf(__('%s by %s', 'latest-posts-grid'), 
                                get_the_date(), 
                                get_the_author()) . 
                         '</div>';
                }
                
                // Post meta information - categories
                echo '<div class="lpg-meta">';
                
                // Post categories
                $categories = get_the_category();
                if (!empty($categories)) {
                    echo '<span class="lpg-categories">';
                    $cat_links = array();
                    foreach ($categories as $category) {
                        $cat_links[] = '<a href="' . esc_url(get_category_link($category->term_id)) . '">' . 
                                      esc_html($category->name) . '</a>';
                    }
                    echo implode(', ', $cat_links);
                    echo '</span>';
                }
                
                echo '</div>'; // End .lpg-meta
                
                // Post excerpt
                echo '<div class="lpg-excerpt">';
                $excerpt = get_the_excerpt();
                echo wp_trim_words($excerpt, 20, '...');
                echo '</div>';
                
                // Read More button
                echo '<a href="' . esc_url(get_permalink()) . '" class="lpg-read-more" aria-label="' . 
                     sprintf(__('Read more about %s', 'latest-posts-grid'), esc_attr($original_title)) . '">' . 
                     __('Read More', 'latest-posts-grid') . '</a>';
                
                echo '</div>'; // End .lpg-grid-item-content
                echo '</div>'; // End .lpg-grid-item
            }
        } else {
            echo '<p class="lpg-no-posts">' . __('No posts found.', 'latest-posts-grid') . '</p>';
        }

        echo '</div>'; // End .lpg-grid
        
        // Add pagination if enabled
        if ($atts['pagination'] && $query->max_num_pages > 1) {
            $this->render_pagination($query, $atts['pagination_type']);
        }
        
        echo '</div>'; // End .lpg-container

        // Reset post data
        wp_reset_postdata();

        // Return the buffered content
        return ob_get_clean();
    }
    
    /**
     * Render pagination links
     * 
     * @param WP_Query $query WP_Query object
     * @param string $pagination_type Type of pagination (numbers, prev_next, both)
     */
    private function render_pagination($query, $pagination_type) {
        if ($query->max_num_pages <= 1) {
            return; // No pagination needed
        }
        
        $current_page = max(1, intval($query->get('paged')));
        $total_pages = intval($query->max_num_pages);
        
        echo '<div class="lpg-pagination" role="navigation" aria-label="' . __('Posts pagination', 'latest-posts-grid') . '">';
        
        if ($pagination_type === 'numbers' || $pagination_type === 'both') {
            // First 3 pages
            for ($i = 1; $i <= min(3, $total_pages); $i++) {
                if ($i === $current_page) {
                    echo '<span class="page-numbers current" aria-current="page">' . $i . '</span>';
                } else {
                    echo '<a href="' . esc_url(get_pagenum_link($i)) . '" class="page-numbers" aria-label="' . 
                         sprintf(__('Go to page %d', 'latest-posts-grid'), $i) . '">' . $i . '</a>';
                }
                
                // Add comma after page number
                if ($i < min(3, $total_pages) || ($total_pages > 3 && $i === 3)) {
                    echo '<span class="page-numbers comma" aria-hidden="true">,</span>';
                }
            }
            
            // Dots and last page
            if ($total_pages > 3) {
                echo '<span class="page-numbers dots" aria-hidden="true">.....</span>';
                
                // Last page
                if ($current_page === $total_pages) {
                    echo '<span class="page-numbers current" aria-current="page">' . $total_pages . '</span>';
                } else {
                    echo '<a href="' . esc_url(get_pagenum_link($total_pages)) . '" class="page-numbers" aria-label="' . 
                         sprintf(__('Go to page %d', 'latest-posts-grid'), $total_pages) . '">' . $total_pages . '</a>';
                }
            }
            
            // Next button (if applicable)
            if ($current_page < $total_pages && ($pagination_type === 'both' || $pagination_type === 'prev_next')) {
                echo '<a href="' . esc_url(get_pagenum_link($current_page + 1)) . '" class="page-numbers next" aria-label="' . 
                     __('Go to next page', 'latest-posts-grid') . '">' . __('Next', 'latest-posts-grid') . ' &rarr;</a>';
            }
        } elseif ($pagination_type === 'prev_next') {
            // Previous link
            if ($current_page > 1) {
                echo '<a href="' . esc_url(get_pagenum_link($current_page - 1)) . '" class="page-numbers prev" aria-label="' . 
                     __('Go to previous page', 'latest-posts-grid') . '">&larr; ' . __('Previous', 'latest-posts-grid') . '</a>';
            }
            
            // Next link
            if ($current_page < $total_pages) {
                echo '<a href="' . esc_url(get_pagenum_link($current_page + 1)) . '" class="page-numbers next" aria-label="' . 
                     __('Go to next page', 'latest-posts-grid') . '">' . __('Next', 'latest-posts-grid') . ' &rarr;</a>';
            }
        }
        
        echo '</div>'; // End .lpg-pagination
    }
    
    /**
     * Create plugin files on activation - SECURE VERSION
     */
    public function activate() {
        // Check if user has proper capabilities
        if (!current_user_can('activate_plugins')) {
            wp_die(__('You do not have sufficient permissions to activate plugins.', 'latest-posts-grid'));
        }
        
        // Create directories securely
        $directories = array(
            LPG_PLUGIN_PATH . 'css',
            LPG_PLUGIN_PATH . 'js'
        );
        
        foreach ($directories as $dir) {
            if (!file_exists($dir)) {
                if (!wp_mkdir_p($dir)) {
                    $this->log_error('Could not create directory: ' . $dir);
                    add_option('lpg_activation_error', sprintf(__('Could not create directory: %s', 'latest-posts-grid'), basename($dir)));
                    return;
                }
            }
        }
        
        // Create files securely with error handling
        $files_to_create = array(
            'css/style.css' => $this->get_frontend_css(),
            'css/admin.css' => $this->get_admin_css(),
            'js/admin.js' => $this->get_admin_js()
        );
        
        foreach ($files_to_create as $file_path => $content) {
            $full_path = LPG_PLUGIN_PATH . $file_path;
            
            if (file_put_contents($full_path, $content) === false) {
                $this->log_error("Could not write file: {$file_path}");
                add_option('lpg_activation_error', sprintf(__('Could not write file: %s', 'latest-posts-grid'), $file_path));
            }
        }
        
        // Set default options
        if (!get_option('lpg_saved_shortcodes')) {
            add_option('lpg_saved_shortcodes', array(), '', 'no');
        }
        
        // Set plugin version
        update_option('lpg_version', $this->version);
        
        // Log successful activation
        $this->log_error('Plugin activated successfully - Version: ' . $this->version);
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up if needed
        $this->log_error('Plugin deactivated');
    }
    
    /**
     * Show admin notices for activation errors
     */
    public function admin_notices() {
        $error = get_option('lpg_activation_error');
        if ($error) {
            echo '<div class="notice notice-error"><p><strong>' . __('Latest Posts Grid:', 'latest-posts-grid') . '</strong> ' . esc_html($error) . '</p></div>';
            delete_option('lpg_activation_error');
        }
    }
    
    /**
     * Add plugin action links
     */
    public function plugin_action_links($links) {
        $plugin_links = array(
            '<a href="' . admin_url('admin.php?page=latest-posts-grid') . '">' . __('Settings', 'latest-posts-grid') . '</a>',
        );
        
        return array_merge($plugin_links, $links);
    }
    
    /**
     * Log errors for debugging
     */
    private function log_error($message, $data = null) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $log_message = 'Latest Posts Grid: ' . $message;
            if ($data) {
                $log_message .= ' Data: ' . print_r($data, true);
            }
            error_log($log_message);
        }
    }
    
    /**
     * Get frontend CSS content
     */
    private function get_frontend_css() {
        return file_get_contents(LPG_PLUGIN_PATH . 'css/style.css');
    }
    
    /**
     * Get admin CSS content 
     */
    private function get_admin_css() {
        return file_get_contents(LPG_PLUGIN_PATH . 'css/admin.css');
    }
    
    /**
     * Get admin JS content
     */
    private function get_admin_js() {
        return file_get_contents(LPG_PLUGIN_PATH . 'js/admin.js');
    }
}

// Initialize the plugin
new Latest_Posts_Grid();